/*
**                ---  selftest.c ---
**
**  SELFTEST requires either a single serial port with a loop-back dongle
**  (connect TX to RX, DTR to DSR, and RTS to CTS), or two serial port
**  on the same computer.
**
**  The program transmits a test string on one port (FirstCOM) and
**  receives on a second port (SecondCOM), where the two ports are
**  connected via a null modem adapter. The received string is tested
**  against the transmit string (they should be idenical).
**
**  Connect the two serial ports (on a single computer) together
**  using a null modem cable. Be sure to modify the configuration
**  section for non-standard PC ports or to setup your multiport
**  board. Note that many multiport boards are either Digiboard or
**  BOCA board compatible.
**
**  IMPORTANT: You may have to modify the port address & IRQ to match
**  your Digiboard or BOCA board installation.
**
**  Compile with /DDPMI for protected mode testing.
*/

#ifdef DPMI
#include "windows.h"
#include "use_dpmi.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <dos.h>
#include <string.h>
#include <conio.h>
#include "pcl4c.h"

#define ESC 0x1b
#define CR  0x0d
#define LF  0x0a

#define READ_PORT(P) inportb(X)

#define PC 1
#define DB 2
#define BB 3
#define CC 4

/*** Global Variables ***/

#define TEST_SIZE 63

static int BaudCode;

char TestSet[TEST_SIZE+1];
int  NbrRuns = 16;
int  TestLength;
int FirstCOM;
int SecondCOM;

int MultiUART = 0;
int MultiIRQ = 0;
int MultiStatus = 0;

void AllDone(void);
void Display(int);
void CheckAlloc(int,char *);
int  ErrorCheck(int);
int  AllocSeg(int);
int  BaudMatch(char *);

static int  RxSeg1 = 0;
static int  TxSeg1 = 0;
static int  RxSeg2 = 0;
static int  TxSeg2 = 0;

char *Ptr[4] = {"SMALL","COMPACT","MEDIUM","LARGE"};

/* standard PC port configuration */

static int StdAdrPC[4] = {0x3f8,0x2f8,0x3e8,0x2e8};
static int StdIrqPC[4] = {4,3,4,3};
static char *BaudRate[10] =  {"300","600","1200","2400","4800","9600",
                              "19200","38400","57600","115200"};
static int Selection = 0;  /* PC, DB, or BB */

void main(int argc, char *argv[])
{int Port;
 char *P;
 int ComLimit = 0;
 char c;
 int Version;
 int Model;
 int i, n, rc;
 int RxBase;
 int TxBase;
 if(argc!=5)
  {printf("Usage: selftest {pc|db|bb|cc} 1stCom 2ndCom baudrate\n");
   exit(1);
  }
 BaudCode = BaudMatch(argv[4]);
 if(BaudCode==-1)
  {printf("Bad baud rate '%s'\n",argv[4]);
   exit(1);
  }
 P = argv[1];
 if((strcmp(P,"pc")==0)||(strcmp(P,"PC")==0)) Selection = PC;
 if((strcmp(P,"db")==0)||(strcmp(P,"DB")==0)) Selection = DB;
 if((strcmp(P,"bb")==0)||(strcmp(P,"BB")==0)) Selection = BB;
 if((strcmp(P,"cc")==0)||(strcmp(P,"CC")==0)) Selection = CC;
 if(Selection==0)
   {puts("Must specify 'PC', 'DB' or 'BB' as 1st argument");
    puts("EG:  SELFTEST PC 1 4");
    exit(1);
   }
 if(Selection==PC) ComLimit = COM4;
 if(Selection==DB) ComLimit = COM4;
 if(Selection==BB) ComLimit = COM4;
 if(Selection==CC) ComLimit = COM4;

 FirstCOM = atoi(argv[2]) -1;
 SecondCOM = atoi(argv[3]) -1;
 printf("FirstCOM  = COM%d\n",1+FirstCOM);
 printf("SecondCOM = COM%d\n",1+SecondCOM);
 if(FirstCOM<COM1)
   {puts("1stCom must be >= COM1");
    exit(1);
   }
 if(SecondCOM>ComLimit)
   {printf("2ndCom must be <= COM%d\n",1+ComLimit);
    exit(1);
   }
 if(Selection==CC)
   {/*** Custom Configuration: 4 port board ***/
    printf("[ Custom configuration: 0x280,288,290,298: IRQ10,15,10,15]\n");
    MultiUART = 0x280; MultiIRQ = IRQ10; MultiStatus = 0;
    SioUART(COM1,0x280); SioIRQ(COM1,IRQ10);
    SioUART(COM2,0x288); SioIRQ(COM2,IRQ15);
    SioUART(COM3,0x290); SioIRQ(COM3,IRQ10);
    SioUART(COM4,0x298); SioIRQ(COM4,IRQ15);
   }
 if(Selection==DB)
   {/*** Custom Configuration: DigiBoard PC/8 ***/
    MultiUART = 0x180; MultiIRQ = IRQ5; MultiStatus = 0x1C0;
    printf("[ Configuring DigiBoard as COM1-COM8 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(1+ComLimit,COM1,MultiStatus,DIGIBOARD);
    for(Port=COM1;Port<=ComLimit;Port++)
      {/* set DigiBoard UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set DigiBoard IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==BB)
   {/*** Custom Configuration: BOCA BB2016 ***/
    MultiUART = 0x100; MultiIRQ = IRQ10; MultiStatus = MultiUART + 7;
    printf("[ Configuring BOCA Board as COM1-COM16 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(1+ComLimit,COM1,MultiStatus,BOCABOARD);
    for(Port=COM1;Port<=ComLimit;Port++)
      {/* set BOCA Board UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set BOCA Board IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==PC)
   {/*** Standard Configuration: 4 port card ***/
    puts("[ Configuring for PC ]");
    for(i=COM1;i<=COM4;i++)
      {SioUART(i,StdAdrPC[i]);
       SioIRQ(i,StdIrqPC[i]);
      }
   }
 CheckAlloc(RxSeg1 = AllocSeg(1024), "RX1");
 ErrorCheck( SioRxBuf(FirstCOM,RxSeg1,Size1024) );
 if(SioInfo('I'))
   {CheckAlloc(TxSeg1 = AllocSeg(1024), "TX1");
    ErrorCheck( SioTxBuf(FirstCOM,TxSeg1,Size1024) );
    ErrorCheck( SioParms(FirstCOM,NoParity,OneStopBit,WordLength8) );
   }
 ErrorCheck( SioParms(FirstCOM,NoParity,OneStopBit,WordLength8) );
 ErrorCheck( SioReset(FirstCOM,BaudCode) );
 printf("COM%d reset\n",1+FirstCOM);
 SioRxClear(FirstCOM);
 SioTxClear(FirstCOM);

 if(FirstCOM!=SecondCOM)
   {/* setup 2nd port (SecondCOM) */
    CheckAlloc(RxSeg2 = AllocSeg(1024), "RX2");
    ErrorCheck( SioRxBuf(SecondCOM,RxSeg2,Size1024) );
    if(SioInfo('I'))
      {CheckAlloc(TxSeg2 = AllocSeg(1024), "TX2");
       ErrorCheck( SioTxBuf(SecondCOM,TxSeg2,Size1024) );
      }
    ErrorCheck( SioParms(SecondCOM,NoParity,OneStopBit,WordLength8) );
    ErrorCheck( SioReset(SecondCOM,BaudCode) );
    printf("COM%d reset\n",1+SecondCOM);
    SioRxClear(SecondCOM);
    SioTxClear(SecondCOM);
   }
 printf("***\n");
 printf("*** SELFTEST 6.0\n");
 printf("***\n");
 Version = SioInfo('V');
 Model = SioInfo('M');
 printf("*** Lib Ver : %d.%d\n",Version/16,Version%16);
 printf("***   Model : %s \n", Ptr[Model&3] );
 printf("*** TX Intr : ");
 if(SioInfo('I')) puts("Enabled.");
 else puts("Disabled.");
 printf("*** DPMI    : ");
 if(SioInfo('P')) printf("YES\n");
 else printf("NO\n");
 /* display port info */
 Display(FirstCOM);
 Display(SecondCOM);
 printf("***\n");

 printf("Starting @ %s baud. COM%d to COM%d\n",argv[4],FirstCOM+1,SecondCOM+1);

 TxBase = SioInfo('T');
 RxBase = SioInfo('R');
 /* build TestSet[] array */
 for(i=0;i<26;i++) TestSet[i] = 'A'+i;
 for(i=0;i<26;i++) TestSet[26+i] = 'a'+i;
 for(i=0;i<10;i++) TestSet[52+i] = '0'+i;
 TestSet[62] = '\n';
 TestSet[63] = '\0';
 printf("Test Set: %s",TestSet);
 /* send test sets */
 printf("\n  Sending set: ");
 for(i=0;i<NbrRuns;i++)
   {/* send test set again */
    printf("%d ",i);
    for(n=0;n<TEST_SIZE;n++)
      {c = TestSet[n];
       SioPutc(FirstCOM,c);
      }
   }
 /* receive test sets */
 printf("\nReceiving set: ");
 for(i=0;i<NbrRuns;i++)
   {/* receive next test set */
    for(n=0;n<TEST_SIZE;n++)
      {rc = SioGetc(SecondCOM,18);
       if(rc<0)
         {printf("\nERROR: ");
          SioError(rc);
          printf("%d bytes in COM%d RX queue\n", SioRxQue(FirstCOM),1+FirstCOM );
          printf("%d bytes in COM%d RX queue\n", SioRxQue(SecondCOM),1+SecondCOM );
          printf("%d RX interrupts\n",SioInfo('R') - RxBase );
          SioDone(FirstCOM);
          if(FirstCOM!=SecondCOM) SioDone(SecondCOM);
          exit(1);
         }
       /* compare character */
       if((char)rc!=TestSet[n])
         {printf("\nERROR: Expecting '%c'(0x%x), received '%c'(0x%x)\n",
            (char)rc,(char)rc,TestSet[n],TestSet[n]);
          SioDone(FirstCOM);
          if(FirstCOM!=SecondCOM) SioDone(SecondCOM);
          exit(1);
         }
      }
    printf("%d ",i);
   } /* end for */
 printf("\n\n");
 /* check RX FIFO */
 TestLength = NbrRuns * TEST_SIZE;
 i = SioInfo('R');
 printf("%3d RX interrupts on %d incoming bytes: ", i-RxBase,TestLength);
 if(i-RxBase<TestLength) puts("RX FIFO is operational");
 else puts("RX FIFO is NOT operational [or not 16550 UART]");
 if(SioInfo('I'))
   {/* check TX FIFO */
    i = SioInfo('T');
    printf("%3d TX interrupts on %d outgoing bytes: ", i-TxBase,TestLength);
    if(i-TxBase<TestLength) puts("TX FIFO is operational");
    else puts("TX FIFO is NOT operational [or not 16550 UART]");
   }
 puts("SUCCESS: Test AOK !");
 AllDone();
 exit(0);
} /* end main */

int ErrorCheck(int Code)
{/* trap PCL error codes */
 if(Code<0)
     {SioError(Code);
      AllDone();
      exit(1);
     }
 return(0);
} /* end ErrorCheck */


void Display(int Port)
{printf("***    COM%d : ",1+Port);
 if(Selection==PC) printf("Adr=%3x IRQ=%d",StdAdrPC[Port],StdIrqPC[Port]);
 else printf("Adr=%3xh IRQ=%d Status=%xh",MultiUART+8*(Port-COM1),MultiIRQ,MultiStatus);
 if( SioFIFO(Port,LEVEL_8) ) printf(" [16550]\n");
 else printf(" [8250/16450]\n");
}


#ifdef DPMI

int AllocSeg(int Size)
{long hMem;
 int  Selector = 0;
 hMem = (long) GlobalDosAlloc( (long)Size );
 if(hMem)
   {/* get selector */
    Selector = LOWORD(hMem);
    GlobalPageLock(Selector);
    return Selector;
   }
 else return 0;
}

#else

int AllocSeg(int Size)
{int Seg;
 char far *Ptr;
 /* allocate far heap */
 Ptr = (char far *) _fmalloc(Size+16);
 if(Ptr==NULL) return 0;
 /* SEG:0 points to buffer */
 Seg = FP_SEG(Ptr) + ((FP_OFF(Ptr)+15)>>4);
 return Seg;
}

#endif

void AllDone(void)
{printf("Shutting down COM%d, ",1+FirstCOM);
 SioDone(FirstCOM);
 printf("COM%d.\n",1+SecondCOM);
 if(FirstCOM!=SecondCOM) SioDone(SecondCOM);
#ifdef DPMI
 printf("Freeing DOS memory: ");
 /* free DOS memory */
 if(RxSeg1)
    {printf("RX1 ");
     GlobalPageUnlock(RxSeg1);
     GlobalDosFree(RxSeg1);
    }
 if(TxSeg1)
    {printf("TX1 ");
     GlobalPageUnlock(TxSeg1);
     GlobalDosFree(TxSeg1);
    }
 if(RxSeg2)
    {printf("RX2 ");
     GlobalPageUnlock(RxSeg2);
     GlobalDosFree(RxSeg2);
    }
 if(TxSeg2)
    {printf("TX2 ");
     GlobalPageUnlock(TxSeg2);
     GlobalDosFree(TxSeg2);
    }
 #endif
 exit(0);
}

void CheckAlloc(int Seg,char *Msg)
{if(Seg==0)
  {printf("Cannot allocate memory for %s\n",Msg);
   AllDone();
   exit(1);
  }
}

int BaudMatch(char *P)
{int i;
 /* find baud rate in table */
 for(i=0;i<10;i++) if(strcmp(BaudRate[i],P)==0) return(i);
 return(-1);
}